/*
 *	LZAri.CPP -- Data Decompression Routines
 *
 *  Adapted by Noose from Original code by Haruhiko Okumura
 *
 *  Revisions:
 *  26/5/98     Corrected cyclical buffer routines to
 *              minimise memory overheads
 *  26/5/98     Fixed huge gaping error which lead to repeat
 *              creations of the class failing
 */

#include <vcl\vcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "lzari.hpp"
#include "Progress.h"

/********** Bit I/O **********/

void Error(char *message)
{
	printf("\n%s\n", message); // Erm, not much use in Windoze
	exit(EXIT_FAILURE);
} // Error(...)

int LZAri::GetBit(void)  /* Get one bit (0 or 1) */
{
	if ((umask >>= 1) == 0) {
		charbuffer = getc(infile);  umask = 128;
	}
	return ((charbuffer & umask) != 0);
} // LZAri::GetBit()

void LZAri::PutBit(int bit)  /* Output one bit (bit = 0,1) */
{
	if (bit) charbuffer |= umask;
	if ((umask >>= 1) == 0) {
		if (putc((char)charbuffer, outfile) == EOF) Error("Write Error");
		charbuffer = 0;  umask = 128;  codesize++;
	}
} // LZAri::PutBit(..)

void LZAri::FlushBitBuffer(void)  /* Send remaining bits */
{
	int  i;
	
	for (i = 0; i < 7; i++) PutBit(0);
} // LZAri::FlushBitBuffer()


/********** Arithmetic Compression **********/

/*  If you are not familiar with arithmetic compression, you should read
		I. E. Witten, R. M. Neal, and J. G. Cleary,
			Communications of the ACM, Vol. 30, pp. 520-540 (1987),
	from which much of this has been borrowed.  */

void LZAri::StartModel(void)  /* Initialize model */
{
	int ch, sym, i;
	
	sym_cum[N_CHAR] = 0;
	for (sym = N_CHAR; sym >= 1; sym--) {
		ch = sym - 1;
		char_to_sym[ch] = sym;  sym_to_char[sym] = ch;
		sym_freq[sym] = 1;
		sym_cum[sym - 1] = sym_cum[sym] + sym_freq[sym];
	}
	sym_freq[0] = 0;  /* sentinel (!= sym_freq[1]) */
	position_cum[N] = 0;
	for (i = N; i >= 1; i--)
		position_cum[i - 1] = position_cum[i] + 10000 / (i + 200);
			/* empirical distribution function (quite tentative) */
			/* Please devise a better mechanism! */
} // LZAri::StartModel()

void LZAri::UpdateModel(int sym)
{
	int i, c, ch_i, ch_sym;
	
	if (sym_cum[0] >= MAX_CUM) {
		c = 0;
		for (i = N_CHAR; i > 0; i--) {
			sym_cum[i] = c;
			c += (sym_freq[i] = (sym_freq[i] + 1) >> 1);
		}
		sym_cum[0] = c;
	}
	for (i = sym; sym_freq[i] == sym_freq[i - 1]; i--) ;
	if (i < sym) {
		ch_i = sym_to_char[i];    ch_sym = sym_to_char[sym];
		sym_to_char[i] = ch_sym;  sym_to_char[sym] = ch_i;
		char_to_sym[ch_i] = sym;  char_to_sym[ch_sym] = i;
	}
	sym_freq[i]++;
	while (--i >= 0) sym_cum[i]++;
} // LZAri::UpdateModel(...)

void LZAri::InitTree(void)  /* Initialize trees */
{
	int  i;

	/* For i = 0 to N - 1, rson[i] and lson[i] will be the right and
	   left children of node i.  These nodes need not be initialized.
	   Also, dad[i] is the parent of node i.  These are initialized to
	   NIL (= N), which stands for 'not used.'
	   For i = 0 to 255, rson[N + i + 1] is the root of the tree
	   for strings that begin with character i.  These are initialized
	   to NIL.  Note there are 256 trees. */

	for (i = N + 1; i <= N + 256; i++) rson[i] = NIL;	/* root */
	for (i = 0; i < N; i++) dad[i] = NIL;	/* node */
} // LZAri::InitTree()

void LZAri::InsertNode(int r)
	/* Inserts string of length F, text_buf[r..r+F-1], into one of the
	   trees (text_buf[r]'th tree) and returns the longest-match position
	   and length via the global variables match_position and match_length.
	   If match_length = F, then removes the old node in favor of the new
	   one, because the old one will be deleted sooner.
	   Note r plays double role, as tree node and position in buffer. */
{
	int  i, p, cmp, temp;
	unsigned char  *key;

	cmp = 1;  key = &text_buf[r];  p = N + 1 + key[0];
	rson[r] = lson[r] = NIL;  match_length = 0;
	for ( ; ; ) {
		if (cmp >= 0) {
			if (rson[p] != NIL) p = rson[p];
			else {  rson[p] = r;  dad[r] = p;  return;  }
		} else {
			if (lson[p] != NIL) p = lson[p];
			else {  lson[p] = r;  dad[r] = p;  return;  }
		}
		for (i = 1; i < F; i++)
			if ((cmp = key[i] - text_buf[p + i]) != 0)  break;
		if (i > THRESHOLD) {
			if (i > match_length) {
				match_position = (r - p) & (N - 1);
				if ((match_length = i) >= F) break;
			} else if (i == match_length) {
				if ((temp = (r - p) & (N - 1)) < match_position)
					match_position = temp;
			}
		}
	}
	dad[r] = dad[p];  lson[r] = lson[p];  rson[r] = rson[p];
	dad[lson[p]] = r;  dad[rson[p]] = r;
	if (rson[dad[p]] == p) rson[dad[p]] = r;
	else                   lson[dad[p]] = r;
	dad[p] = NIL;  /* remove p */
} // LZAri::InsertNode(..)

void LZAri::DeleteNode(int p)  /* Delete node p from tree */
{
	int  q;
	
	if (dad[p] == NIL) return;  /* not in tree */
	if (rson[p] == NIL) q = lson[p];
	else if (lson[p] == NIL) q = rson[p];
	else {
		q = lson[p];
		if (rson[q] != NIL) {
			do {  q = rson[q];  } while (rson[q] != NIL);
			rson[dad[q]] = lson[q];  dad[lson[q]] = dad[q];
			lson[q] = lson[p];  dad[lson[p]] = q;
		}
		rson[q] = rson[p];  dad[rson[p]] = q;
	}
	dad[q] = dad[p];
	if (rson[dad[p]] == p) rson[dad[p]] = q;
	else                   lson[dad[p]] = q;
	dad[p] = NIL;
} // LZAri::DeleteNode(..)

void LZAri::StartDecode(void)
{
	int i;

	for (i = 0; i < M + 2; i++)
		value = 2 * value + GetBit();
} // LZAri::StartDecode()

int LZAri::DecodeChar(void)
{
	int	 sym, ch;
	unsigned long int  range;
	
	range = high - low;
	sym = BinarySearchSym((unsigned int)
		(((value - low + 1) * sym_cum[0] - 1) / range));
	high = low + (range * sym_cum[sym - 1]) / sym_cum[0];
	low +=       (range * sym_cum[sym    ]) / sym_cum[0];
	for ( ; ; ) {
		if (low >= Q2) {
			value -= Q2;  low -= Q2;  high -= Q2;
		} else if (low >= Q1 && high <= Q3) {
			value -= Q1;  low -= Q1;  high -= Q1;
		} else if (high > Q2) break;
		low += low;  high += high;
		value = 2 * value + GetBit();
	}
	ch = sym_to_char[sym];
	UpdateModel(sym);
	return ch;
} // LZAri::DecodeChar()

int LZAri::DecodePosition(void)
{
	int position;
	unsigned long int  range;
	
	range = high - low;
	position = BinarySearchPos((unsigned int)
		(((value - low + 1) * position_cum[0] - 1) / range));
	high = low + (range * position_cum[position    ]) / position_cum[0];
	low +=       (range * position_cum[position + 1]) / position_cum[0];
	for ( ; ; ) {
		if (low >= Q2) {
			value -= Q2;  low -= Q2;  high -= Q2;
		} else if (low >= Q1 && high <= Q3) {
			value -= Q1;  low -= Q1;  high -= Q1;
		} else if (high > Q2) break;
		low += low;  high += high;
		value = 2 * value + GetBit();
	}
	return position;
} // LZAri::DecodePosition()

int LZAri::BinarySearchSym(unsigned int x)
	/* 1      if x >= sym_cum[1],
	   N_CHAR if sym_cum[N_CHAR] > x,
	   i such that sym_cum[i - 1] > x >= sym_cum[i] otherwise */
{
	int i, j, k;
	
	i = 1;  j = N_CHAR;
	while (i < j) {
		k = (i + j) / 2;
		if (sym_cum[k] > x) i = k + 1;  else j = k;
	}
	return i;
} // LZAri::BinarySearchSym(...)

int LZAri::BinarySearchPos(unsigned int x)
	/* 0 if x >= position_cum[1],
	   N - 1 if position_cum[N] > x,
	   i such that position_cum[i] > x >= position_cum[i + 1] otherwise */
{
	int i, j, k;
	
	i = 1;  j = N;
	while (i < j) {
		k = (i + j) / 2;
		if (position_cum[k] > x) i = k + 1;  else j = k;
	}
	return i - 1;
} // LZAri::BinarySearchPos(...)


void LZAri::PutChar(int c) {
    if (RBOffset==MAX_READ_BUFF)
        RBOffset=0;
    ReadBuffer[RBOffset++]=(char)c;
    ReadBuffer[RBOffset]='\0';
} // LZAri::PutChar(...);

void LZAri::Output(int bit)  /* Output 1 bit, followed by its complements */
{
	PutBit(bit);
	for ( ; shifts > 0; shifts--) PutBit(! bit);
} // LZAri::Output(..)

void LZAri::EncodeChar(int ch)
{
	int  sym;
	unsigned long int  range;

	sym = char_to_sym[ch];
	range = high - low;
	high = low + (range * sym_cum[sym - 1]) / sym_cum[0];
	low +=       (range * sym_cum[sym    ]) / sym_cum[0];
	for ( ; ; ) {
		if (high <= Q2) Output(0);
		else if (low >= Q2) {
			Output(1);  low -= Q2;  high -= Q2;
		} else if (low >= Q1 && high <= Q3) {
			shifts++;  low -= Q1;  high -= Q1;
		} else break;
		low += low;  high += high;
	}
	UpdateModel(sym);
} // LZAri::EncodeChar(..)

void LZAri::EncodePosition(int position)
{
	unsigned long int  range;

	range = high - low;
	high = low + (range * position_cum[position    ]) / position_cum[0];
	low +=       (range * position_cum[position + 1]) / position_cum[0];
	for ( ; ; ) {
		if (high <= Q2) Output(0);
		else if (low >= Q2) {
			Output(1);  low -= Q2;  high -= Q2;
		} else if (low >= Q1 && high <= Q3) {
			shifts++;  low -= Q1;  high -= Q1;
		} else break;
		low += low;  high += high;
	}
} // LZAri::EncodePosition(..)

void LZAri::EncodeEnd(void)
{
	shifts++;
	if (low < Q1) Output(0);  else Output(1);
	FlushBitBuffer();  /* flush bits remaining in buffer */
} // LZAri::EncodeEnd(..)

char LZAri::GetChar(void) {
    if (OutputBuffer[RBOffset]!='\0')
        return (OutputBuffer[RBOffset++]);
    else
        return (OutputBuffer[RBOffset]);
} // LZAri::GetChar(void)

int LZAri::UpdateProgress(void) {
	Counter++;
	if (Counter>=Step) {
 		Counter=0;
        return TRUE;
	}
   	return FALSE;
}

void LZAri::Encode(void)
{
	int  i, len, s, last_match_length;
    char c;

	Working=new TWorking(Application);
    Working->Show();

	s = 0;  r = N - F;
	for (i = s; i < r; i++) text_buf[i] = ' ';
	for (len = 0; len < F && (c = GetChar()) != '\0'; len++)
		text_buf[r + len] = c;

	textsize = len;
	for (i = 1; i <= F; i++) InsertNode(r - i);
	InsertNode(r);

	do {
		if (match_length > len) match_length = len;
		if (match_length <= THRESHOLD) {
			match_length = 1;  EncodeChar(text_buf[r]);
		} else {
			EncodeChar(255 - THRESHOLD + match_length);
			EncodePosition(match_position - 1);
		}
		last_match_length = match_length;
		for (i = 0; i < last_match_length &&
				(c = GetChar()) != '\0'; i++) {
			DeleteNode(s);  text_buf[s] = c;
			if (s < F - 1) text_buf[s + N] = c;
			s = (s + 1) & (N - 1);
			r = (r + 1) & (N - 1);
			InsertNode(r);
            if (UpdateProgress())
            	Working->ProgressBar->StepIt();
		}
		if ((textsize += i) > printcount)
		    printcount += 1024;
		while (i++ < last_match_length) {
			DeleteNode(s);
			s = (s + 1) & (N - 1);
			r = (r + 1) & (N - 1);
			if (--len) InsertNode(r);
		}
	} while (len > 0);
	EncodeEnd();
    delete (Working);
} // LZAri::Encode()

/*
 * LZAri::Decode(void)
 *
 * Decodes a single block (or byte) from the input stream.
 *
 *  In: Nothing
 * Out: Nothing
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 */
void LZAri::Decode(void)
{
	int  i, j, k, c;

	c = DecodeChar();
	if (c < 256) {
		PutChar(c);  text_buf[r++] = (char)c;
		r &= (N - 1);  count++;
	} else {
		i = (r - DecodePosition() - 1) & (N - 1);
		j = c - 255 + THRESHOLD;
		for (k = 0; k < j; k++) {
			c = text_buf[(i + k) & (N - 1)];
			PutChar(c);  text_buf[r++] = (char)c;
			r &= (N - 1);  count++;
		}
	}
	if (count > printcount)
		printcount += 1024;
} // LZAri::Decode()

/*
 * LZAri::eof()
 *
 * Returns whether the file is at it's end yet.
 *
 *  In: Nothing
 * Out: integer     -   Boolean Flag
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 */
int LZAri::eof() {
    return _eofFlag;
} // LZAri::eof()

int LZAri::CompleteBuffer(int *ai,int *aIndex,char *Buffer) {
    int EndFound,i,Index;

    EndFound=FALSE;

    Index=*aIndex;
    i=*ai;

    while ( (i!=RBOffset) && (EndFound==FALSE) ) {
        if (i==MAX_READ_BUFF)
            i=0;
        Buffer[Index]=ReadBuffer[i];
        if (ReadBuffer[i]=='\x0a')
            EndFound=TRUE;
        Index++; i++;
    }

    *ai=i;
    *aIndex=Index;

    return EndFound;
}

/*
 * LZAri::Readln(char *Buffer)
 * 
 * Reads a single line from the compressed source file.
 * The routine creates a cyclical buffer which holds any chars
 * which have been "read-ahead" because of the compression.
 * The cyclical buffer should be larger than the length of the
 * longest line to prevent overlap.
 *
 *  In: Buffer[]    -   Where to store the line data
 * Out: Nothing
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 *            29/5/98 Noose - Fixed the stupid CompleteBuffer function
 */


void LZAri::Readln(char *Buffer) {
    int Index,i,EndFound;
    
    ReadBuffer[MAX_READ_BUFF]='\0'; // Ensures ReadBuffer[MAX_READ_BUFF]!=10

    Index=0;                    // Output buffer index
    EndFound=FALSE;             // Not found the end yet
    i=RBLineStart;              // Point to the start of the current line

    while ( (count<textsize) && (EndFound==FALSE) ) {
        if (i==RBOffset)        // Have we run out of data in the buffer?
            Decode();           // Decode another block
        if (count<textsize)     // Are we still before the end of file?
            EndFound=CompleteBuffer(&i,&Index,Buffer);
        else
            _eofFlag=TRUE;      // It's the end of the file
    }

    if (_eofFlag) 
        CompleteBuffer(&i,&Index,Buffer);

    RBLineStart=i;          // No more text in cyclical buffer
    Buffer[Index]='\0';         // Terminate string
} // LZAri::Readln(...)

/*
 * LZAri::LZAri(char *fn)
 *
 * Constructor function for the LZAri class.
 * Initialises all the variables for the decompression and
 * opens the input file specified by fn.
 *
 *  In: fn          -   FileName to load
 * Out: Nothing
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 */
LZAri::LZAri(char *fn,char *aBuffer) {
    int i;
    char HeadBuffer[5],ID_Text[]="SCR";

    umask=0;
    infile=NULL;
    textsize = 0; codesize = 0; printcount = 0;

    low = 0; high = Q4; value = 0;
    shifts = 0;  // counts for magnifying low and high around Q2

    if (aBuffer==NULL) {  // It's a decode call
        infile=fopen(fn,"rb");
        fread(HeadBuffer,3,1,infile);
        HeadBuffer[3]='\0';
        if (strcmp(HeadBuffer,ID_Text))
            Error("Not a valid SCR file.");

    	if (fread(&textsize, sizeof textsize, 1, infile) < 1)
    		Error("Read Error");  /* read size of text */
    	if (textsize == 0) return;
    	StartDecode();  StartModel();
    	for (i = 0; i < N - F; i++) text_buf[i] = ' ';
    	r = N - F;
        RBOffset=0;
        RBLineStart=0;
        count=0;
        for (i=0;i<MAX_READ_BUFF;i++)
            ReadBuffer[i]='\0';
        _eofFlag=FALSE;
    } else { // It must be an encode call
    	charbuffer = 0;
    	umask = 128;

        textsize=strlen(aBuffer);
        Step=textsize/100;
        Counter=0;
        OutputBuffer=aBuffer;

        outfile = fopen(fn, "wb");
        RBOffset=0;

        if (fwrite(ID_Text,3,1,outfile) < 1)
            Error("Write Error");   // Output identifier
    	if (fwrite(&textsize, sizeof textsize, 1, outfile) < 1)
    		Error("Write Error");   // output size of text (when decompressed)

    	codesize += sizeof textsize;
    	if (textsize == 0) return;

    	textsize = 0;
    	StartModel();  InitTree();
        Encode();
        fclose(outfile);
    }
} // LZAri::LZAri(...)

/*
 * LZAri::~LZAri()
 *
 * Deconstuctor function for the LZAri Class.
 * Closes the open file.
 */
LZAri::~LZAri() {
	if (infile!=NULL) {
	    fclose(infile);             // Close the input file
 	    infile=NULL;
	}
} // LZAri::~LZAri()

